#!/bin/bash

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -r, --runningversion    <release version>        runnning release version
  -i, --installingversion <release version>        installing release version
  -d, --directory         <path>                   path to persistent directory to update
  -h, --help                                       display help
EOM

	exit 2
}

function delete_redis_key () {

  # convert long options to getopt short ones
  for arg in "$@"; do
    shift
    case "$arg" in
      "--port")                set -- "$@" "-p" ;;
      "--key")                 set -- "$@" "-k" ;;
      *)                       set -- "$@" "$arg"
    esac
  done

  cursor="0"
  pattern=""
  port=""
  CLI_BIN="/usr/bin/redis-cli"

  OPTIND=1         # Reset in case getopts has been used previously in the shell.
  while getopts "p:k:" opt; do
    case "$opt" in
      p)  port=$OPTARG
        ;;
      k)  pattern=$OPTARG
        ;;
      esac
  done
  shift $((OPTIND-1))

  while
  mapfile -t matches < <(${CLI_BIN} -c -p ${port} SCAN $cursor MATCH "${pattern}")
  cursor=${matches[0]}
  [[ "${cursor}" -ne "0" ]]
  do
    #Scan shows "(empty array)" on the value paired with cursor.
    #Therefore, we treat the result as and array.
    mapfile -t result < <(echo "${matches[1]}")
    for i in "${!result[@]}"; do
      match=${result[$i]}
      if [[ "${match}" == *"${pattern}" ]]; then
        output="$(${CLI_BIN} -c -p ${port} DEL ${match})"
        echo "Deleted: ${match}"
        if [[ $output -ne 1 ]]; then
          echo "Failed to delete ${match}"
        fi
      fi
    done
  done
}

export -f delete_redis_key

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--runningversion")      set -- "$@" "-r" ;;
    "--installingversion")   set -- "$@" "-i" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

RUNNING_RELEASE_VERSION=''
INSTALLING_RELEASE_VERSION=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?r:i:d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    r)  RUNNING_RELEASE_VERSION=$OPTARG
      ;;
    i)  INSTALLING_RELEASE_VERSION=$OPTARG
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done


[ "${1:-}" = "--" ] && shift

if [ -z $RUNNING_RELEASE_VERSION ]
then
  LOG "Failed to get the running release version"
  exit 1
fi

if [ -z $INSTALLING_RELEASE_VERSION ]
then
  LOG "Failed to get the installing release version"
  exit 1
fi

if [ ! -d "$PERSISTENT_DIRECTORY" ]
then
  LOG "The directory $PERSISTENT_DIRECTORY does not exist."
  exit 1
else
  LOG "Upgrade script will update the persistent content at : $PERSISTENT_DIRECTORY."
fi

if [[ -f /proc/device-tree/dcpm40csozrplus ]]
then
  THROW 1 "Release $INSTALLING_RELEASE_VERSION does not support this hardware."
fi

export RUNNING_RELEASE_VERSION
export INSTALLING_RELEASE_VERSION
SCRIPTS_DIR=$(readlink -f $0 | xargs dirname)
MIGRATE_TO_REDIS=0
if [[ "$RUNNING_RELEASE_VERSION" == "$INSTALLING_RELEASE_VERSION" ]]
then
  LOG "Running version and installing version are the same."
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  REDIS_PORT=30001 START_REDIS=false ${SCRIPTS_DIR}/copy-nw-settings-to-redis.sh
  REDIS_PORT=30001 ${SCRIPTS_DIR}/dcp-upgrades/dcp404-path-upgrade.sh
elif [[ "$RUNNING_RELEASE_VERSION" =~ "6.0." ]]
then
  LOG "Doing upgrade from $RUNNING_RELEASE_VERSION"
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  MIGRATE_TO_REDIS=1
elif [[ "$RUNNING_RELEASE_VERSION" =~ "6.1." ]]
then
  LOG "Doing upgrade from $RUNNING_RELEASE_VERSION"
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  MIGRATE_TO_REDIS=1
elif [[ "$RUNNING_RELEASE_VERSION" == "6.0.2" ]]
then
  LOG "Doing upgrade from 6.0.2"
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
elif [[ "$RUNNING_RELEASE_VERSION" == "6.1.1" ]]
then
  LOG "Doing upgrade from 6.1.1"
  ${SCRIPTS_DIR}/update-hostname.sh -d $PERSISTENT_DIRECTORY
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
# Note! Not all upgrade scripts below should be executed when upgrading FROM release 10 and forward. For example "upgrade_dcpr_cloop_wss_ports.sh" MUST NOT be run when upgrading from release 10.
elif ([[ "$RUNNING_RELEASE_VERSION" =~ "7." ]] || [[ "$RUNNING_RELEASE_VERSION" =~ "8." ]] ||
      [[ "$RUNNING_RELEASE_VERSION" =~ ^9\. ]]) && [[ "$INSTALLING_RELEASE_VERSION" =~ ^10\. ]]
then
  LOG "Doing upgrade from $RUNNING_RELEASE_VERSION to $INSTALLING_RELEASE_VERSION"
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY 
  REDIS_PORT=30001 START_REDIS=false ${SCRIPTS_DIR}/copy-nw-settings-to-redis.sh
  REDIS_PORT=30001 ${SCRIPTS_DIR}/dcp-upgrades/dcp404-path-upgrade.sh
  ret=$? 
  if [ $ret -ne 0 ]; then 
    LOG "Script failed: dcp404-path-upgrade.sh, code: $ret"
    exit 1 
  fi
  ${SCRIPTS_DIR}/upgrade_lldp.sh 
  ${SCRIPTS_DIR}/upgrade_dcpr_cloop_wss_ports.sh
  ret=$?
  if [ $ret -ne 0 ]; then
    LOG "Script failed: upgrade_dcpr_cloop_wss_ports.sh, code: $ret"
    exit 1
  fi
  ${SCRIPTS_DIR}/upgrade_dcp2_automation_mode.sh

# This is a special case for a customer that started out with M32 before cloop v2 was introduced
elif [[ "$RUNNING_RELEASE_VERSION" =~ ^10\.2 ]] && 
  ([[ "$INSTALLING_RELEASE_VERSION" =~ ^9[0-9]{2,}.[0-9]{2,}.[0-9]+ ]] || [[ "$INSTALLING_RELEASE_VERSION" =~ ^12\. ]])
then
  LOG "Upgrading from $RUNNING_RELEASE_VERSION to $INSTALLING_RELEASE_VERSION. Migrating values from old cloop to cloop v2. "
  ${SCRIPTS_DIR}/dcp-upgrades/cloop_gen1_to_gen2_upgrade.sh
  ret=$?
  if [ $ret -ne 0 ]; then
    LOG "Script failed: cloop_gen1_to_gen2_upgrade.sh, code: $ret"
    exit 1
  fi
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  REDIS_PORT=30001 START_REDIS=false ${SCRIPTS_DIR}/copy-nw-settings-to-redis.sh
  ${SCRIPTS_DIR}/upgrade_lldp.sh
  ${SCRIPTS_DIR}/dcp-upgrades/cloop_frequency_index_m32.sh $RUNNING_RELEASE_VERSION
  ret=$?
  if [ $ret -ne 0 ]; then
    LOG "Script failed: cloop_frequency_index_m32.sh, code: $ret"
    exit 1
  fi
  LOG "Migration complete to $INSTALLING_RELEASE_VERSION complete."

elif ( [[ "$RUNNING_RELEASE_VERSION" =~ ^9\. ]] || [[ "$RUNNING_RELEASE_VERSION" =~ ^10\. ]] ) &&
 ([[ "$INSTALLING_RELEASE_VERSION" =~ ^1[12]\. ]] || [[ "$INSTALLING_RELEASE_VERSION" =~ ^9[0-9]{2,}.[0-9]{2,}.[0-9]+ ]])
then
  LOG "Doing upgrade from $RUNNING_RELEASE_VERSION to $INSTALLING_RELEASE_VERSION"
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  REDIS_PORT=30001 START_REDIS=false UPGRADE=true ${SCRIPTS_DIR}/copy-nw-settings-to-redis.sh
  ${SCRIPTS_DIR}/upgrade_dcp2_automation_mode.sh
  if [[ "$RUNNING_RELEASE_VERSION" =~ ^9\. ]]
  then
    ${SCRIPTS_DIR}/upgrade_dcpr_cloop_wss_ports.sh
    ret=$?
    if [ $ret -ne 0 ]; then
      LOG "Script failed: upgrade_dcpr_cloop_wss_ports.sh, code: $ret"
      exit 1
    fi
  fi
  ${SCRIPTS_DIR}/dcp-upgrades/upgrade_cloops_flexgrid.sh
  ret=$?
  if [ $ret -ne 0 ]; then
    LOG "Script failed: upgrade_cloops_flexgrid.sh, code: $ret"
    exit 1
  fi
elif [[ "$RUNNING_RELEASE_VERSION" =~ ^11\. ]] && 
    ([[ "$INSTALLING_RELEASE_VERSION" =~ ^9[0-9]{2,}.[0-9]{2,}.[0-9]+ ]] || [[ "$INSTALLING_RELEASE_VERSION" =~ ^12\. ]])
then
  LOG "Doing upgrade from $RUNNING_RELEASE_VERSION to $INSTALLING_RELEASE_VERSION"
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  REDIS_PORT=30001 START_REDIS=false UPGRADE=true ${SCRIPTS_DIR}/copy-nw-settings-to-redis.sh
  REDIS_PORT=30001 ${SCRIPTS_DIR}/dcp-upgrades/dcp404-path-upgrade.sh
  ${SCRIPTS_DIR}/upgrade_lldp.sh
  ${SCRIPTS_DIR}/upgrade_dcp2_automation_mode.sh
  if [[ "$RUNNING_RELEASE_VERSION" =~ ^11.2\. ]]
  then
    LOG "Doing an extra set of edfa adminStatus for DCP-2 because default values are changed between 11.2 and later"
    ${SCRIPTS_DIR}/upgrade_changed_default_edfa_admin_status.sh
    ret=$?
    if [ $ret -ne 0 ]; then
      LOG "Script failed: upgrade_changed_default_edfa_admin_status.sh, code: $ret"
      exit 1
    fi
    LOG "Doing an extra set of automationmode for DCP-2 and DCP-M32 because default values are changed between 11.2 and later"
    ${SCRIPTS_DIR}/upgrade_changed_default_automation_mode.sh
    ret=$?
    if [ $ret -ne 0 ]; then
      LOG "Script failed: upgrade_changed_default_automation_mode.sh, code: $ret"
      exit 1
    fi
  fi
  if [[ "$RUNNING_RELEASE_VERSION" =~ ^11\.2\.1 ]]; then 
    ${SCRIPTS_DIR}/dcp-upgrades/cloop_frequency_index_m32.sh $RUNNING_RELEASE_VERSION
    ret=$?
    if [ $ret -ne 0 ]; then
      LOG "Script failed: cloop_frequency_index_m32.sh, code: $ret"
      exit 1
    fi
  fi
elif [[ "$RUNNING_RELEASE_VERSION" =~ ^9[0-9]{2,}.[0-9]{2,}.[0-9]+ ]]
  then
  LOG "Doing upgrade from $RUNNING_RELEASE_VERSION to $INSTALLING_RELEASE_VERSION"
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  REDIS_PORT=30001 START_REDIS=false ${SCRIPTS_DIR}/copy-nw-settings-to-redis.sh
  REDIS_PORT=30001 ${SCRIPTS_DIR}/dcp-upgrades/dcp404-path-upgrade.sh
  ${SCRIPTS_DIR}/upgrade_lldp.sh
  ${SCRIPTS_DIR}/upgrade_dcp2_automation_mode.sh
else
  ${SCRIPTS_DIR}/update-network-interfaces.sh -d $PERSISTENT_DIRECTORY
  LOG "Do not know upgrade path"
fi

 REDIS_PORT=30001 ${SCRIPTS_DIR}/remove-password-in-redis.sh
 ${SCRIPTS_DIR}/update-tacacs-radius.sh -d $PERSISTENT_DIRECTORY
 ${SCRIPTS_DIR}/update-syslog-ng-config.sh -d $PERSISTENT_DIRECTORY
${SCRIPTS_DIR}/upgrade_dcpr_tunnel.sh

# Check if smartmanager is available or not
[[ -f /usr/sbin/smartmanager/smartmanager.js ]] && MIGRATE_TO_REDIS=1

if [[ ${MIGRATE_TO_REDIS} -ne 0 ]]; then
  THROW 1 "Upgrade not supported. Please first upgrade to an earlier release before proceeding to $INSTALLING_RELEASE_VERSION."
else
   REDIS_PORT=30001 ${SCRIPTS_DIR}/handle-shadow-user-accounts.sh
fi

exit 0
