#!/bin/bash

function usage {
    cat <<EOM
Usage: $(basename "$0") [OPTION]...

  -d, --directory    <path>        path to persistent directory to update
  -h, --help                       display help
EOM

  exit 2
}

# convert long options to getopt short ones
for arg in "$@"; do
  shift
  case "$arg" in
    "--help")                set -- "$@" "-h" ;;
    "--directory")           set -- "$@" "-d" ;;
    *)                       set -- "$@" "$arg"
  esac
done

PERSISTENT_DIRECTORY=''

# A POSIX variable
OPTIND=1         # Reset in case getopts has been used previously in the shell.

while getopts "h?d:" opt; do
  case "$opt" in
    h|\?)
      usage
      exit 0
      ;;
    d)  PERSISTENT_DIRECTORY=$OPTARG
      ;;
    esac
done

shift $((OPTIND-1))

[ "${1:-}" = "--" ] && shift

if [ ! -d "$PERSISTENT_DIRECTORY" ]
then
  echo "The directory $PERSISTENT_DIRECTORY does not exist."
  exit 1
fi

tacacs=$(grep -c "auth[[:blank:]]\\+include[[:blank:]]\\+tacacs" /var/persistent/etc/pam.d/sshd)
radius=$(grep -c "pam_radius_auth.so" /var/persistent/etc/pam.d/sshd)
PAM_RADIUS_AUTH_SO="/usr/lib/security/pam_radius_auth.so"
RADIUS_FLAG="[success=done new_authtok_reqd=done default=bad authinfo_unavail=ignore module_unknown=ignore ignore=ignore]"
RADIUS_AUTH="auth $RADIUS_FLAG $PAM_RADIUS_AUTH_SO"
RADIUS_ACCOUNT="account $RADIUS_FLAG $PAM_RADIUS_AUTH_SO"
RADIUS_SESSION="session $RADIUS_FLAG $PAM_RADIUS_AUTH_SO"

radius_user_disabled=$(grep -c radius_user:\! $PERSISTENT_DIRECTORY/etc/shadow)
radius_user_bk_disabled=$(grep -c radius_user:\! $PERSISTENT_DIRECTORY/etc/shadow-)
if [ ${radius_user_disabled} == 0 ]; then
  sed -i "s|^\(radius_user:\)\(.*:\)\(.*:.*:.*:.*:.*:.*:\)|\1\!\2\3|g" $PERSISTENT_DIRECTORY/etc/shadow
fi
if [ ${radius_user_bk_disabled} == 0 ]; then
  sed -i "s|^\(radius_user:\)\(.*:\)\(.*:.*:.*:.*:.*:.*:\)|\1\!\2\3|g" $PERSISTENT_DIRECTORY/etc/shadow-
fi

if [ ${tacacs} != 0 ]; then
  if ! grep -q -E "auth +include +tacacs" /var/persistent/etc/pam.d/sudo; then
    sed -i "1iauth        include tacacs\\naccount     include tacacs\\nsession     include tacacs" $PERSISTENT_DIRECTORY/etc/pam.d/sudo
  fi
elif [ ${radius} != 0 ]; then
  if ! grep -q "pam_radius_auth.so" /var/persistent/etc/pam.d/sudo; then
    sed -i "1i $RADIUS_AUTH \\n$RADIUS_ACCOUNT \\n$RADIUS_SESSION" $PERSISTENT_DIRECTORY/etc/pam.d/sudo
  fi
fi

sed -i 's/#*timeout=.*/timeout=2/g' $PERSISTENT_DIRECTORY/etc/tacplus_nss.conf
if ! grep -q "retry=" $PERSISTENT_DIRECTORY/etc/tacplus_nss.conf; then
  sed -i -e '$a\\n# Should not take time for retry to connect to tacplus server\nretry=0' $PERSISTENT_DIRECTORY/etc/tacplus_nss.conf
fi

if ! grep -q "fallback_user=" $PERSISTENT_DIRECTORY/etc/tacplus_nss.conf; then
  sed -i -e '$a\\n# Account for fallback login with local account when server is unavailable.\nfallback_user=admin' $PERSISTENT_DIRECTORY/etc/tacplus_nss.conf
fi

sed -i 's/admin/radius_priv_user/g' $PERSISTENT_DIRECTORY/etc/nss_mapuser.conf
sed -i 's/admin/radius_priv_user/g' $PERSISTENT_DIRECTORY/etc/raddb/server

# Revert sshd_config and login.defs to default, only update login timeout.
BASEDIR=$(dirname $0)
if [ -f $BASEDIR/../../image/factory_defaults/etc/login.defs ]; then
  cp $BASEDIR/../../image/factory_defaults/etc/login.defs $PERSISTENT_DIRECTORY/etc/
fi
if [[ ${radius} != 0 || ${tacacs} != 0 ]]; then
  sed -i -E 's/#*LOGIN_TIMEOUT\s*[0-9]+.*/LOGIN_TIMEOUT           780/g' $PERSISTENT_DIRECTORY/etc/login.defs
fi

if [ -f $BASEDIR/../../image/factory_defaults/etc/ssh/sshd_config ]; then
  cp $BASEDIR/../../image/factory_defaults/etc/ssh/sshd_config $PERSISTENT_DIRECTORY/etc/ssh/
fi
if [[ ${radius} != 0 || ${tacacs} != 0 ]]; then
  sed -i -E 's/#*LoginGraceTime\s*[0-9]+.*/LoginGraceTime 13m/g' $PERSISTENT_DIRECTORY/etc/ssh/sshd_config
fi

if [ -f $BASEDIR/../../image/factory_defaults/etc/tacplus_nss.conf ]; then
  new_tacplus_exclude_user=$(grep -E "exclude_users=*,*" $BASEDIR/../../image/factory_defaults/etc/tacplus_nss.conf)
  sed -i "s/#*exclude_users=.*/$new_tacplus_exclude_user/g" $PERSISTENT_DIRECTORY/etc/tacplus_nss.conf
fi

if [ -f $BASEDIR/../../image/factory_defaults/etc/nss_mapuser.conf ]; then
  new_radius_exclude_user=$(grep -E "exclude_users=*,*" $BASEDIR/../../image/factory_defaults/etc/nss_mapuser.conf )
  sed -i "s/#*exclude_users=.*/$new_radius_exclude_user/g" $PERSISTENT_DIRECTORY/etc/nss_mapuser.conf
fi
