-- KEYS[1] = cluster prefix
-- ARGV[1] = product
-- USAGE: return "Failed:" as a keyword used to indicate that the script failed to upgrade.sh. Then continue with the error message.


local product = ARGV[1]
local tag = KEYS[1]


if product ~= "DCP-R-9D-CS" and product ~= "DCP-R-34D-CS" then
  return "Failed: Invalid product " .. product
end

local channel_plan = {}

local function convert_wss_port_name(wss_port_name) 
  if product == "DCP-R-9D-CS" then
    if wss_port_name == "localAD" then
      return { err = nil, port = 2 }
    elseif wss_port_name == "xc1" then
      return { err = nil, port = 4 }
    elseif wss_port_name == "xc2" then
      return { err = nil, port = 6 }
    elseif wss_port_name == "xc3" then
      return { err = nil, port = 8 }
    elseif wss_port_name == "xc4Wss1" then
      return { err = nil, port = 1 }
    elseif wss_port_name == "xc4Wss2" then
      return { err = nil, port = 3 }
    elseif wss_port_name == "xc4Wss3" then
      return { err = nil, port = 5 }
    elseif wss_port_name == "xc4Wss4" then
      return { err = nil, port = 7 }
    elseif wss_port_name == "xc4Wss5" then
      return { err = nil, port = 9 }
    else 
      return { err = "Failed: Invalid WSS port name " .. wss_port_name }
    end
  elseif product == "DCP-R-34D-CS" then
    if wss_port_name == "off" then
      return { err = nil, port = 100 }
    end
    local port_number = tonumber(wss_port_name:match("xc(%d+)"))
    if port_number then
      return { err = nil, port = port_number }
    else 
      return { err = "Failed: Invalid WSS port name " .. wss_port_name }
    end
  else
    return { err = "Failed: Invalid product " .. product }
  end
end

local function extract_value(json_string)
  if type(json_string) ~= "string" then
    return { err = "Failed: invalid json string ", value = json_string }
  end

  local decoded = cjson.decode(json_string)
  local json_value = decoded.value
  if type(json_value) == "number" then
    -- If we have a number. We need to convert it to string with 2 decimal points for decimal safety
    json_value = string.format("%.2f", json_value)
  end
  return { err = nil, value = json_value }
end

local function set_value_to_new_path(json_object, new_path, device_handler)
  redis.pcall('DEL', tag .. ":" .. device_handler .. ":" .. new_path) -- Delete just in case
  local output = redis.pcall('DH.CREATE_CONFIG_KEY', tag, device_handler, new_path, json_object)
  if type(output) == "table" and output.err then
    return { err = "Failed: Failed to create key for " .. new_path }
  end
  output = redis.pcall('DH.SET', tag .. ":" .. device_handler .. ":" .. new_path, json_object)
  if type(output) == "table" and output.err then
    return { err = "Failed: Failed to set value for " .. new_path }
  end
  return { err = nil }
end

local function get_channel_description(channel)
  local channel_freq = 9130 + (channel * 10)
  local channel_description = redis.pcall('dh.get', tag .. ":so/interface:" .. channel_freq .. "/config/description")
  if type(channel_description) == "table" and channel_description.err then
    return { err = "Failed: Failed to get channel description for channel " .. channel_freq }
  end
  local description = extract_value(channel_description)
  if description.err then
    return { err = description.err }
  end
  return { err = nil, description = description.value }
end

local function get_wss_add_attenuation(channel)
  if product == "DCP-R-9D-CS" then
    local wss_index = channel * 2
    local wss_add_attenuation = redis.pcall('dh.get', tag .. ":dh/wss:config/module/add/channel/" .. wss_index .. "/attenuation")
    if type(wss_add_attenuation) == "table" and wss_add_attenuation.err then
      return { err = "Failed: Failed to get wssAddAttenuation for rx channel " .. channel }
    end
    return { err = nil, value = wss_add_attenuation }
  elseif product == "DCP-R-34D-CS" then
    local wss_add_attenuation = redis.pcall('dh.get', tag .. ":dh/wss:config/module/add/channel/" .. channel .. "/attenuation")
    if type(wss_add_attenuation) == "table" and wss_add_attenuation.err then
      return { err = "Failed: Failed to get wssAddAttenuation for rx channel " .. channel }
    end
    return { err = nil, value = wss_add_attenuation }
  else
    return { err = "Failed: Invalid product " .. product }
  end
end

-- Script starts here
local target_output_power_redis = redis.pcall('dh.get', tag .. ":dh/cloopcrx:config/system/setGlobalTargetOutputPower")
if type(target_output_power_redis) == "table" and target_output_power_redis.err then
  return "Failed: Failed to get targetOutputPower"
end
local targetOutputPower = extract_value(target_output_power_redis)
if targetOutputPower.err then
  return "Failed: Failed to extract targetOutputPower"
end
local targetOutputPower_json = "{\"value\":" .. targetOutputPower.value .. "}"

-- targetOutputPower recreate
local result = set_value_to_new_path(targetOutputPower_json, "config/client/rx/targetOutputPower", "dh/cloop")
if result.err then
  return result.err
end

-- OPoutOffset recreate
local opout_offset_redis = redis.pcall('dh.get', tag .. ":dh/cloopl:config/line/1/OPoutOffset")
if type(opout_offset_redis) == "table" and opout_offset_redis.err then
  return "Failed: Failed to get OPoutOffset"
end
local opoutOffset = extract_value(opout_offset_redis)
if opoutOffset.err then
  return "Failed: Failed to extract OPoutOffset"
end

if type(opoutOffset.value) == "userdata" then
  -- In case of opoutOffset not being configured, the value will be considered as user data by lua.
  -- User data has to be converted before being added to the json below.
  -- For a more general approach, we set the value to 0.0 so that we can avoid conversions
  opoutOffset.value = 0.0
elseif opoutOffset.value == nil then
  opoutOffset.value = 0.0
end

local opoutOffset_json = "{\"value\":" .. opoutOffset.value .. "}"
result = set_value_to_new_path(opoutOffset_json, "config/line/OPoutOffset", "so/cloop")
if result.err then
  return result.err
end

-- Booster gain recreate
local booster_gain_redis = redis.pcall('dh.get', tag .. ":dh/cloopl:config/line/1/booster/gain")
if type(booster_gain_redis) == "table" and booster_gain_redis.err then
  return "Failed: Failed to get boosterGain"
end
local boosterGain = extract_value(booster_gain_redis)
if boosterGain.err then
  return "Failed: Failed to extract boosterGain"
end
local boosterGain_json = "{\"value\":" .. boosterGain.value .. "}"
result = set_value_to_new_path(boosterGain_json, "config/gain", "dh/booster")
if result.err then
  return result.err
end

for channel = 1, 48 do
  local opticalControlMode_redis = redis.pcall('dh.get', tag .. ":dh/cloopcrx:config/client/" .. channel .. "/rx/opticalControlMode");
  if type(opticalControlMode_redis) == "table" and opticalControlMode_redis.err then
    return "Failed: Failed to get opticalControlMode for rx channel " .. channel
  end

  local opticalControlMode = extract_value(opticalControlMode_redis)
  if opticalControlMode.err then
    return "Failed: Failed to extract opticalControlMode for rx channel " .. channel
  end
  opticalControlMode = opticalControlMode.value
  local channel_element = {}
  local center_frequency = 19140000 + ((channel - 1) * 10000)
  local port = 99;
  local width = 10000

  if opticalControlMode ~= "off" and opticalControlMode ~= "power" then
    result = redis.pcall('dh.get', tag .. ":dh/cloop:config/client/" .. channel .. "/wssPortName")
    if type(result) == "table" and result.err then
      return "Failed: WSS portName not found for channel " .. channel
    else
      local port_extract = extract_value(result)
      if port_extract.err then
        return "Failed: Failed to extract WSS port name for channel " .. channel
      end
      port = convert_wss_port_name(port_extract.value)
      if port.err then
        return port.err .. " for channel " .. channel
      end
    end
    
    channel_element['center_frequency'] = center_frequency
    channel_element['port'] = port.port
    channel_element['width'] = width
    channel_element['version'] = "10.0.1"
    table.insert(channel_plan, channel_element)

    --opticalControlMode recreate
    local opticalControlMode_json = "{\"value\":\"" .. opticalControlMode .. "\"}"
    result = set_value_to_new_path(opticalControlMode_json, "config/client/" .. center_frequency .. "/rx/opticalControlMode", "dh/cloop")
    if result.err then
      return result.err
    end

    result = set_value_to_new_path(opticalControlMode_json, "config/client/" .. center_frequency .. "/tx/opticalControlMode", "dh/cloop")
    if result.err then
      return result.err
    end

    --wssDropAttenuation recreate
    result = redis.pcall('DH.GET', tag .. ":dh/cloopctx:config/client/" .. channel .. "/tx/attenuation")
    if type(result) == "table" and result.err then
      return "Failed: Failed to get wssDropAttenuation for tx channel " .. channel
    end
    local wssDropAttenuation = extract_value(result)
    if wssDropAttenuation.err then
      return "Failed: Failed to extract wssDropAttenuation for tx channel " .. channel
    end
    local wssDropAttenuation_json = "{\"value\":" .. wssDropAttenuation.value .. "}"
    
    result = set_value_to_new_path(wssDropAttenuation_json, "config/client/" .. center_frequency .. "/tx/defaultAttenuation", "dh/cloop")
    if result.err then
      return result.err
    end

    --wssAddAttenuation recreate
    local wssAddAttenuation_redis = get_wss_add_attenuation(channel)
    if wssAddAttenuation_redis.err then
      return "Failed: Failed to get wssAddAttenuation for rx channel " .. channel
    end
    local wssAddAttenuation = extract_value(wssAddAttenuation_redis.value)
    if wssAddAttenuation.err then
      return "Failed: Failed to extract wssAddAttenuation for rx channel " .. channel
    end

    local wssAddAttenuation_json = "{\"value\":" .. wssAddAttenuation.value .. "}"

    result = set_value_to_new_path(wssAddAttenuation_json, "config/module/add/channel/" .. center_frequency .. "/attenuation", "dh/wss")
    if result.err then
      return result.err
    end

    --Channel description recreate
    result = get_channel_description(channel)
    if type(result) == "table" and result.err then
      return "Failed: " .. result.err
    end
    local channel_description = result.description
    result = set_value_to_new_path("{\"value\":\"" .. channel_description .. "\"}", "config/client/" .. center_frequency .. "/rx/description", "dh/cloop")
    if result.err then
      return result.err
    end
  elseif opticalControlMode == "power" then
    return "Failed: Please put all active channels from opticalControlMode 'power' to 'gainLoss' before upgrading"
  end
end

local channel_plan_json = {}
if next(channel_plan) == nil then
  channel_plan_json['value'] = "[]"
else
  channel_plan_json['value'] = channel_plan
end
--redis.call('SET', tag .. ":so/cloop:channel/plan/rx", cjson.encode(channel_plan_json))
--redis.call('SET', tag .. ":so/cloop:channel/plan/tx", cjson.encode(channel_plan_json))
result = set_value_to_new_path(cjson.encode(channel_plan_json), "channel/plan/rx", "so/cloop")
if result.err then
  return result.err
end

result = set_value_to_new_path(cjson.encode(channel_plan_json), "channel/plan/tx", "so/cloop")
if result.err then
  return result.err
end

return "Success: Channel plan updated successfully"