#!/bin/bash

#
# Script to migrate cloop client configuration from release 10.2.1 and 11.2.1
#
# pre release 12 we had channel path index as sequential numbers, i.e. client/1/, client/2/ etc.
# post release 12 we have channel path index as frequency, i.e. client/19140000/, client/19155000/ etc.
#
# this script will list cloop config paths from redis, update the index and re-create them as new attributes.
#
# release 10.2.1 runs cloop g1, so we expect cloop_gen1_to_gen2_upgrade.sh to be run before to migrate the config
# from cloop g1 to cloop g2, aligning it with the 11.2.1 config format.
#

if [[ ! -f "/proc/device-tree/dcpm32csozrplus" ]]; then
  exit 0
fi

echo "Running $0 on $1"

CHANNEL_START_FREQ=19140000
CHANNEL_WIDTH=15000

REDIS_PORT=30001
FROM_VERSION=$1

CONFIGS=$(redis-cli -p $REDIS_PORT keys *dh/cloop:*config*)
for OLD_PATH in $CONFIGS; do
  VALUE=$(redis-cli -p $REDIS_PORT DH.GET $OLD_PATH)
  if [[ $? -ne 0 ]] || [[ ! $VALUE ]]; then
    echo "Failed to get value for $OLD_PATH"
    continue
  fi
  NEW_PATH=$OLD_PATH
  if [[ $(echo $OLD_PATH | grep -E "client/[[:alnum:]]+/") ]]; then
    CLIENT_NO=$(echo $OLD_PATH | sed -E 's/.+client\/([[:alnum:]]+)\/.+/\1/')
    if [[ $CLIENT_NO -gt 32 ]]; then
      continue
    fi
    NEW_INDEX=$(($CHANNEL_START_FREQ + $(($CLIENT_NO - 1)) * $CHANNEL_WIDTH))
    NEW_PATH=$(echo $OLD_PATH | sed -e "s/\/$CLIENT_NO\//\/$NEW_INDEX\//g")
  fi
  if [[ $NEW_PATH != $OLD_PATH ]]; then
    PATH_TAIL=$(echo $NEW_PATH | sed -e "s/.*:dh\/cloop://g")
    redis-cli -p $REDIS_PORT DH.CREATE_CONFIG_KEY $REDIS_CONF_HASH_TAG dh/cloop $PATH_TAIL "$VALUE"
    redis-cli -p $REDIS_PORT DH.SET ${REDIS_CONF_HASH_TAG}:dh/cloop:${PATH_TAIL} "$VALUE"
    if [[ $? -ne 0 ]]; then
      echo "Failed to migrate key $OLD_PATH to $NEW_PATH"
    else
      redis-cli -p $REDIS_PORT DEL $OLD_PATH
      echo "Migrated $OLD_PATH to $NEW_PATH, value $VALUE"
    fi
  else
    echo "No migration needed for $OLD_PATH"
  fi
done

echo "Finished running $0"

exit 0